<?php
// bot.php - Khushi Bot (Complete Version)
require_once 'config.php';
$update = json_decode(file_get_contents("php://input"), true);
if (!$update) exit;

// Globals
$apiURL = "https://api.telegram.org/bot".BOT_TOKEN."/";

// Utilities
function isAdminId($user_id) {
    $admins = json_decode(file_get_contents(ADMINS_FILE), true);
    return in_array(intval($user_id), $admins);
}

function isGroupAdmin($user_id, $chat_id) {
    global $apiURL;
    $url = $apiURL."getChatMember?chat_id=$chat_id&user_id=$user_id";
    $result = json_decode(file_get_contents($url), true);
    if ($result['ok']) {
        $status = $result['result']['status'];
        return in_array($status, ['administrator', 'creator']);
    }
    return false;
}

function sendMessage($chat_id, $text, $reply_to=null, $parse_mode='HTML') {
    global $apiURL;
    $params = ['chat_id'=>$chat_id,'text'=>$text,'parse_mode'=>$parse_mode];
    if ($reply_to) $params['reply_to_message_id'] = $reply_to;
    $url = $apiURL."sendMessage?".http_build_query($params);
    return file_get_contents($url);
}

function deleteMessage($chat_id, $message_id) {
    global $apiURL;
    return @file_get_contents($apiURL."deleteMessage?chat_id=$chat_id&message_id=$message_id");
}

function sendPoll($chat_id, $question, $options, $is_anonymous=false) {
    global $apiURL;
    $opts = json_encode(array_values($options));
    $params = ['chat_id'=>$chat_id,'question'=>$question,'options'=>$opts,'is_anonymous'=>$is_anonymous];
    return file_get_contents($apiURL."sendPoll?".http_build_query($params));
}

function banUser($chat_id, $user_id) {
    global $apiURL;
    return @file_get_contents($apiURL."banChatMember?chat_id=$chat_id&user_id=$user_id");
}

function unbanUser($chat_id, $user_id) {
    global $apiURL;
    return @file_get_contents($apiURL."unbanChatMember?chat_id=$chat_id&user_id=$user_id");
}

function restrictUser($chat_id, $user_id, $until_date=null) {
    global $apiURL;
    $params = [
        'chat_id' => $chat_id,
        'user_id' => $user_id,
        'permissions' => json_encode(['can_send_messages' => false])
    ];
    if ($until_date) $params['until_date'] = $until_date;
    return @file_get_contents($apiURL."restrictChatMember?".http_build_query($params));
}

function unrestrictUser($chat_id, $user_id) {
    global $apiURL;
    $params = [
        'chat_id' => $chat_id,
        'user_id' => $user_id,
        'permissions' => json_encode([
            'can_send_messages' => true,
            'can_send_media_messages' => true,
            'can_send_polls' => true,
            'can_send_other_messages' => true,
            'can_add_web_page_previews' => true
        ])
    ];
    return @file_get_contents($apiURL."restrictChatMember?".http_build_query($params));
}

function log_event($type, $meta='') {
    global $conn;
    $stmt = $conn->prepare("INSERT INTO logs (event_type, meta) VALUES (?, ?)");
    $stmt->bind_param("ss", $type, $meta);
    $stmt->execute();
    $stmt->close();
}

// Extract user ID from argument or reply
function extractUserId($arg, $msg) {
    // If replying to a message
    if (isset($msg['reply_to_message']['from']['id'])) {
        return $msg['reply_to_message']['from']['id'];
    }
    
    // If argument is a user ID
    if (is_numeric($arg)) {
        return intval($arg);
    }
    
    // If argument is username (without @)
    if (preg_match('/^@?[a-zA-Z0-9_]+$/', $arg)) {
        // In a real scenario, you'd need to get user_id from username
        // For simplicity, we'll return null here
        return null;
    }
    
    return null;
}

// Update handling
// 1) message updates
if (isset($update['message'])) {
    $msg = $update['message'];
    $chat_id = $msg['chat']['id'];
    $from = $msg['from'] ?? [];
    $user_id = $from['id'] ?? 0;
    $username = $from['username'] ?? '';
    $first = $from['first_name'] ?? '';
    $last = $from['last_name'] ?? '';
    $message_id = $msg['message_id'];
    $text = $msg['text'] ?? '';
    $chat_type = $msg['chat']['type'];

    // save user to users table / update last_seen
    $stmt = $conn->prepare("INSERT INTO users (user_id, username, first_name, last_name, first_seen, last_seen) VALUES (?, ?, ?, ?, NOW(), NOW()) ON DUPLICATE KEY UPDATE username=VALUES(username), first_name=VALUES(first_name), last_name=VALUES(last_name), last_seen=NOW()");
    $stmt->bind_param("isss", $user_id, $username, $first, $last);
    $stmt->execute();
    $stmt->close();

    // Welcome / Goodbye - new_chat_members / left_chat_member
    if (isset($msg['new_chat_members'])) {
        foreach ($msg['new_chat_members'] as $new) {
            $welcome = "Welcome, <a href='tg://user?id=".$new['id']."'>".$new['first_name']."</a>!\nPlease follow the group rules.";
            sendMessage($chat_id, $welcome);
            log_event('welcome', json_encode($new));
        }
    }
    if (isset($msg['left_chat_member'])) {
        $left = $msg['left_chat_member'];
        $bye = "Goodbye, ".$left['first_name']."! We will miss you.";
        sendMessage($chat_id, $bye);
        log_event('goodbye', json_encode($left));
    }

    // === LINK REMOVER === 
    // Auto link remover: if message contains http/https/www. and sender is NOT admin => delete
    if (preg_match('/(https?:\/\/|www\.|\.(com|org|net|in|co|tk|ml|ga|cf|xyz|top|site|online|live|store|blog|tech|info|io|ai|ru|cn|uk|de|fr|es|it|nl|eu|biz|shop|app|dev|fun|club|news|pro|tv|me|pw|work|space|website|stream|support|agency|cloud|digital|guru|expert|services|solutions|systems|technology|ventures|vision|world|zone|helioho\.st))/i', $text)) {
        $is_admin = isGroupAdmin($user_id, $chat_id) || isAdminId($user_id);
        if (!$is_admin) {
            // delete message and optionally warn
            deleteMessage($chat_id, $message_id);
            sendMessage($chat_id, "❌ <b>Links are allowed only for admins!</b>\nYour message has been removed.", $message_id);
            log_event('link_removed', "user:$user_id chat:$chat_id");
            exit;
        }
    }

    // Check Blocked
    $res = $conn->query("SELECT * FROM blocked_users WHERE user_id=".intval($user_id));
    if ($res && $res->num_rows>0) {
        // silently ignore or send notice
        sendMessage($chat_id, "You are blocked from using this bot.");
        exit;
    }
    // Check Muted: if muted in DB then delete messages until unmute
    $r = $conn->query("SELECT * FROM muted_users WHERE user_id=".intval($user_id));
    if ($r && $r->num_rows>0) {
        $row = $r->fetch_assoc();
        $mute_until = $row['mute_until'];
        if ($mute_until === null || strtotime($mute_until) > time()) {
            // delete message and inform
            deleteMessage($chat_id, $message_id);
            sendMessage($chat_id, "You are muted and cannot send messages now.");
            exit;
        } else {
            // unmute expired
            $conn->query("DELETE FROM muted_users WHERE user_id=".intval($user_id));
        }
    }

    // === Keyword Auto Reply (From DB) ===
    if (!empty($text)) {
        $sql = "SELECT * FROM keywords";
        $result = $conn->query($sql);
        while ($row = $result->fetch_assoc()) {
            if (stripos($text, $row["keyword"]) !== false) {
                sendMessage($chat_id, $row["reply"]);
                log_event('keyword_reply', $row["keyword"]);
                exit;
            }
        }
    }

    // === Commands (admin and users) ===
    if (strpos($text, '/') === 0) {
        $parts = explode(' ', $text, 2);
        $cmd = strtolower($parts[0]);
        $arg = trim($parts[1] ?? '');

        // === POLL COMMAND (Fixed) ===
        if ($cmd === '/poll') {
            $is_admin = isGroupAdmin($user_id, $chat_id) || isAdminId($user_id);
            if (!$is_admin) {
                sendMessage($chat_id, "❌ Only admins can create polls!", $message_id);
                exit;
            }
            
            if (empty($arg)) {
                sendMessage($chat_id, "📊 <b>Poll Creation</b>\n\nUsage: <code>/poll Question | Option1 | Option2 | Option3</code>\n\nExample:\n<code>/poll Best Color? | Red | Blue | Green</code>", $message_id);
                exit;
            }
            
            // Split by | but ignore spaces around |
            $poll_parts = array_map('trim', explode('|', $arg));
            $question = $poll_parts[0] ?? '';
            $options = array_slice($poll_parts, 1);
            
            if (empty($question) || count($options) < 2) {
                sendMessage($chat_id, "❌ <b>Invalid poll format!</b>\n\nYou need:\n• 1 Question\n• At least 2 Options\n\nFormat: <code>/poll Question | Option1 | Option2 | Option3</code>", $message_id);
                exit;
            }
            
            if (count($options) > 10) {
                sendMessage($chat_id, "❌ <b>Too many options!</b>\nMaximum 10 options allowed.", $message_id);
                exit;
            }
            
            sendPoll($chat_id, $question, $options);
            log_event('create_poll', $question);
            exit;
        }

        // === BAN COMMAND ===
        if ($cmd === '/ban') {
            $is_admin = isGroupAdmin($user_id, $chat_id) || isAdminId($user_id);
            if (!$is_admin) {
                sendMessage($chat_id, "❌ Only admins can ban users!", $message_id);
                exit;
            }
            
            $target_id = extractUserId($arg, $msg);
            if (!$target_id) {
                sendMessage($chat_id, "❌ <b>Usage:</b> Reply to a user or use: <code>/ban @username</code> or <code>/ban user_id</code>", $message_id);
                exit;
            }
            
            if (isGroupAdmin($target_id, $chat_id)) {
                sendMessage($chat_id, "❌ Cannot ban another admin!", $message_id);
                exit;
            }
            
            $result = banUser($chat_id, $target_id);
            $res = json_decode($result, true);
            if ($res['ok']) {
                sendMessage($chat_id, "✅ User has been banned!", $message_id);
                log_event('cmd_ban', $target_id);
            } else {
                sendMessage($chat_id, "❌ Failed to ban user!", $message_id);
            }
            exit;
        }

        // === UNBAN COMMAND ===
        if ($cmd === '/unban') {
            $is_admin = isGroupAdmin($user_id, $chat_id) || isAdminId($user_id);
            if (!$is_admin) {
                sendMessage($chat_id, "❌ Only admins can unban users!", $message_id);
                exit;
            }
            
            $target_id = extractUserId($arg, $msg);
            if (!$target_id) {
                sendMessage($chat_id, "❌ <b>Usage:</b> <code>/unban user_id</code>", $message_id);
                exit;
            }
            
            $result = unbanUser($chat_id, $target_id);
            $res = json_decode($result, true);
            if ($res['ok']) {
                sendMessage($chat_id, "✅ User has been unbanned!", $message_id);
                log_event('cmd_unban', $target_id);
            } else {
                sendMessage($chat_id, "❌ Failed to unban user!", $message_id);
            }
            exit;
        }

        // === MUTE COMMAND ===
        if ($cmd === '/mute') {
            $is_admin = isGroupAdmin($user_id, $chat_id) || isAdminId($user_id);
            if (!$is_admin) {
                sendMessage($chat_id, "❌ Only admins can mute users!", $message_id);
                exit;
            }
            
            $target_id = extractUserId($arg, $msg);
            if (!$target_id) {
                sendMessage($chat_id, "❌ <b>Usage:</b> Reply to a user or use: <code>/mute @username</code> or <code>/mute user_id</code>", $message_id);
                exit;
            }
            
            if (isGroupAdmin($target_id, $chat_id)) {
                sendMessage($chat_id, "❌ Cannot mute another admin!", $message_id);
                exit;
            }
            
            $result = restrictUser($chat_id, $target_id, time() + 3600); // 1 hour mute
            $res = json_decode($result, true);
            if ($res['ok']) {
                sendMessage($chat_id, "✅ User has been muted for 1 hour!", $message_id);
                log_event('cmd_mute', "$target_id for 60 min");
            } else {
                sendMessage($chat_id, "❌ Failed to mute user!", $message_id);
            }
            exit;
        }

        // === UNMUTE COMMAND ===
        if ($cmd === '/unmute') {
            $is_admin = isGroupAdmin($user_id, $chat_id) || isAdminId($user_id);
            if (!$is_admin) {
                sendMessage($chat_id, "❌ Only admins can unmute users!", $message_id);
                exit;
            }
            
            $target_id = extractUserId($arg, $msg);
            if (!$target_id) {
                sendMessage($chat_id, "❌ <b>Usage:</b> <code>/unmute user_id</code>", $message_id);
                exit;
            }
            
            $result = unrestrictUser($chat_id, $target_id);
            $res = json_decode($result, true);
            if ($res['ok']) {
                sendMessage($chat_id, "✅ User has been unmuted!", $message_id);
                log_event('cmd_unmute', $target_id);
            } else {
                sendMessage($chat_id, "❌ Failed to unmute user!", $message_id);
            }
            exit;
        }

        // /block <user_id> - admin only
        if ($cmd === '/block' && isAdminId($user_id)) {
            $target = intval($arg);
            if ($target) {
                $stmt = $conn->prepare("INSERT INTO blocked_users (user_id, reason) VALUES (?, '') ON DUPLICATE KEY UPDATE blocked_at=NOW()");
                $stmt->bind_param("i",$target);
                $stmt->execute();
                $stmt->close();
                sendMessage($chat_id, "User $target blocked.");
                log_event('cmd_block', $target);
            } else sendMessage($chat_id, "Usage: /block <user_id>");
            exit;
        }

        if ($cmd === '/unblock' && isAdminId($user_id)) {
            $target = intval($arg);
            if ($target) {
                $stmt = $conn->prepare("DELETE FROM blocked_users WHERE user_id=?");
                $stmt->bind_param("i",$target);
                $stmt->execute();
                $stmt->close();
                sendMessage($chat_id, "User $target unblocked.");
                log_event('cmd_unblock', $target);
            } else sendMessage($chat_id, "Usage: /unblock <user_id>");
            exit;
        }

        // /mute <user_id> <minutes>
        if ($cmd === '/mute' && isAdminId($user_id)) {
            $a = preg_split('/\s+/', $arg);
            $target = intval($a[0] ?? 0);
            $mins = intval($a[1] ?? 60);
            if ($target) {
                $until = date("Y-m-d H:i:s", time() + max(0,$mins)*60);
                $stmt = $conn->prepare("INSERT INTO muted_users (user_id, mute_until) VALUES (?, ?) ON DUPLICATE KEY UPDATE mute_until=VALUES(mute_until)");
                $stmt->bind_param("is",$target,$until);
                $stmt->execute(); $stmt->close();
                sendMessage($chat_id, "User $target muted for $mins minutes.");
                log_event('cmd_mute', "$target for $mins min");
            } else sendMessage($chat_id, "Usage: /mute <user_id> <minutes>");
            exit;
        }
        if ($cmd === '/unmute' && isAdminId($user_id)) {
            $target = intval($arg);
            if ($target) {
                $stmt = $conn->prepare("DELETE FROM muted_users WHERE user_id=?");
                $stmt->bind_param("i",$target);
                $stmt->execute(); $stmt->close();
                sendMessage($chat_id, "User $target unmuted.");
                log_event('cmd_unmute', $target);
            } else sendMessage($chat_id, "Usage: /unmute <user_id>");
            exit;
        }

        // /status - bot status
        if ($cmd === '/status') {
            $u = $conn->query("SELECT COUNT(*) as c FROM users")->fetch_assoc()['c'];
            $blk = $conn->query("SELECT COUNT(*) as c FROM blocked_users")->fetch_assoc()['c'];
            $mut = $conn->query("SELECT COUNT(*) as c FROM muted_users")->fetch_assoc()['c'];
            sendMessage($chat_id, "🤖 <b>Khushi Bot Status</b>\n\nUsers: $u\nBlocked: $blk\nMuted: $mut");
            exit;
        }

        // /viewusers - show recent users (admin)
        if ($cmd === '/viewusers' && isAdminId($user_id)) {
            $res = $conn->query("SELECT user_id,username,first_name,last_name,last_seen FROM users ORDER BY last_seen DESC LIMIT 50");
            $txt = "Recent users:\n";
            while ($r = $res->fetch_assoc()) {
                $txt .= $r['user_id']." - ".($r['username']?:$r['first_name'])." (".substr($r['last_seen'],0,19).")\n";
            }
            sendMessage($chat_id, $txt);
            exit;
        }

        // /help
        if ($cmd === '/help' || $cmd === '/start') {
            $help = "🤖 <b>Khushi Bot</b>\n\n";
            
            if (in_array($chat_type, ['group', 'supergroup'])) {
                $help .= "👮 <b>Admin Commands:</b>\n";
                $help .= "📊 /poll - Create a poll (Question | Option1 | Option2)\n";
                $help .= "🔨 /ban - Ban a user\n";
                $help .= "🔓 /unban - Unban a user\n";
                $help .= "🔇 /mute - Mute a user (1 hour)\n";
                $help .= "🔊 /unmute - Unmute a user\n\n";
                $help .= "🔗 <b>Auto Features:</b>\n";
                $help .= "• Link removal for non-admins\n";
                $help .= "• Admin-only link posting\n";
            }
            
            $help .= "📝 <b>Note:</b> All admin commands work only in groups and only for group admins.";
            
            sendMessage($chat_id, $help, $message_id);
            exit;
        }
    }

    // === Gemini AI reply (if not other handlers matched) ===
    if (!empty($text)) {
        // If message length small and not command, call Gemini
        if (strlen($text) < 1000) {
            // build prompt
            $reply = "समझ नहीं आया 😅";
            if (defined('GEMINI_KEY') && GEMINI_KEY) {
                // Ask Gemini
                $gk = GEMINI_KEY;
                $url = "https://generativelanguage.googleapis.com/v1beta/models/gemini-2.0-flash:generateContent?key=$gk";
                $prompt = "You are chatting in a Telegram group as a human friend. User said: \"$text\" Reply in the same language (Hindi if Hindi, English if English). Keep it short, natural, friendly, and casual. Avoid sounding like a bot or AI. Use emojis sometimes.";
                $data = [ "contents"=>[ ["parts"=>[["text"=>$prompt]]] ] ];
                $ch = curl_init($url);
                curl_setopt($ch, CURLOPT_HTTPHEADER, ["Content-Type: application/json"]);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_POST, true);
                curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
                $response = curl_exec($ch);
                curl_close($ch);
                $res = json_decode($response, true);
                if (isset($res["candidates"][0]["content"]["parts"][0]["text"])) {
                    $reply = $res["candidates"][0]["content"]["parts"][0]["text"];
                } else {
                    // fallback simple echo
                    $reply = "मैंने सुना: ".$text;
                }
            } else {
                $reply = "Gemini key not configured. Set GEMINI_KEY in config.";
            }
            sendMessage($chat_id, $reply, $message_id);
            exit;
        }
    }

    exit;
}

// 2) chat_member updates (welcome/leave)
if (isset($update['chat_member'])) {
    $cm = $update['chat_member'];
    $chat = $cm['chat'];
    $old = $cm['old_chat_member'];
    $new = $cm['new_chat_member'];
    $chat_id = $chat['id'];

    // member became member or left
    if ($old['status'] === 'left' && in_array($new['status'], ['member','administrator','creator'])) {
        $name = $new['user']['first_name'] ?? '';
        sendMessage($chat_id, "Welcome $name to ".$chat['title']);
        log_event('welcome_chat_member', json_encode($new));
    } elseif (in_array($old['status'], ['member','administrator','creator']) && $new['status'] === 'left') {
        $name = $new['user']['first_name'] ?? '';
        sendMessage($chat_id, "$name has left the chat.");
        log_event('left_chat_member', json_encode($new));
    }
    exit;
}

// 3) other updates (ignored)
http_response_code(200);
?>